from fleming.system_test.fle_sys import *


# Alpha Measurement Test Functions
#=====================================================================================    
# High Level Functions to import in GroundControl: predefined Tasks
# 2022-02-08/Kay Struebing
# https://confluence-ext.perkinelmer.com/display/HamRD/Fleming+Alpha+Laser+Tests

# Global Constants
PATH, SCRIPT = os.path.split(__file__)
BASE_NAME = SCRIPT.split('.')[0]
ADJUSTMENT_PATH = '/home/pkiuser/.config/pyrunner/'

async def alpha_laser_temp_test(start = 0.1, stop = 0.5, power = 3103, delay_s = 20.0, step = 0.01):
    # Tested 2022-02-08 on Newport /kstruebing
    TEST_NAME = 'alpha_laser_temp_test'
    file_name = get_datafile(PATH, BASE_NAME, TEST_NAME)
    time_s = datetime.now().strftime('%H:%M:%S')
    with open(file_name, 'a') as data:
        data.write(f'Data Output from  {BASE_NAME}.{TEST_NAME}\n')
        data.write(f'Temperature Control Range: {start}..{stop}\n')
        data.write(f'Alpha Laser Power: {power}\n')
        data.write(f'Start: {time_s}\n')
        data.write(f'sample;temp;pd;\n')
    steps = int((stop - start) / step)
    await al_enable()
    await al_setpower(power)
    temp = start
    for s in range(steps):      
        await al_settemp(temp)
        await al_on()
        await asyncio.sleep(delay_s)
        print(f'{s}: {temp}: {await al_gettemp()}')
        await write_alpha(file_name, s)         
        await al_off()
        temp = start + s * step
    return(f'{TEST_NAME} finished, Data written to {file_name}')
    
async def alpha_laser_power_test(start = 0, stop = 3500, temp = 0.35, delay_s = 1.0, step = 5.0):
    # Tested 2022-02-08 on Newport /kstruebing
    TEST_NAME = 'alpha_laser_power_test'
    file_name = get_datafile(PATH, BASE_NAME, TEST_NAME)
    time_s = datetime.now().strftime('%H:%M:%S')
    start_delay = 20.0
    with open(file_name, 'a') as data:
        data.write(f"Data Output from  {BASE_NAME}.{TEST_NAME}\n")
        data.write(f"Temperature Control Setpoint: {temp}\n")
        data.write(f'Alpha Laser Power Range: {start}..{stop}\n')        
        data.write(f'Start: {time_s}\n')
        data.write(f'sample;temp;pd;\n')
    steps = int((stop - start) / step)
    await al_enable()
    await al_settemp(temp)
    await asyncio.sleep(start_delay)
    power = start
    for s in range(steps):        
        await al_setpower(power)
        await al_on()
        await asyncio.sleep(delay_s)
        print(f'{s}: {power}: {await al_getpd()}')
        await write_alpha(file_name, s) 
        await al_off()
        power = start + s * step
    return(f'{TEST_NAME} finished, Data written to {file_name}')

async def alpha_laser_temp_response(setpoint, power, duration_s):
    # Tested 2022-06-29/kstruebing
    # Step response for alpha laser temperature control

    TEST_NAME = 'alpha_laser_temp_response'
    file_name = get_datafile(PATH, BASE_NAME, TEST_NAME)
    start_time = datetime.now().strftime('%H:%M:%S')
    data = []
    els = VUnits.instance.hal.excitationLightSelector
    alpha_laser_block_position = 113.8
    
    # Goto beam blocking position
    els.Move(alpha_laser_block_position)
    
    # Write header to datafile
    with open(file_name, 'a') as file:
        file.write(f"Data Output from  {BASE_NAME}.{TEST_NAME}\n")
        file.write(f"        Test started: {start_time}\n")
        file.write(f"Temperature Setpoint: {setpoint}\n")
        file.write(f'               Power: {power}\n')
        file.write(f'sample;temp;pd;\n')
    
    # Start measurement 
    (data, pulse_result) = await asyncio.gather(record_al_data(duration_s), alpha_pulse(duration_s, power, setpoint))
    
    # write to screen
    for line in data:
        print(line)
    
    with open(file_name, 'a') as f:
        for line in data:
            f.write(line)
     
    return(f'{TEST_NAME}: {len(data)} samples written to {file_name}')


async def alpha_pulse(duration_s, power, setpoint):
    result = f"alpha_pulse: Executed"
    
    try:

        await al_enable()
        await al_setpower(power)
        await al_settemp(setpoint)
        await asyncio.sleep(duration_s/4)
        await al_on()
        await asyncio.sleep(duration_s/2)
        await al_off()
        await asyncio.sleep(duration_s/4)

    except Exception as ex:
        result = f"Error occured, alpha_pulse: {str(ex)}"
        logger.error(result)

    return(result)

async def record_al_data(duration_s):
    eef = VUnits.instance.hal.nodes['EEFNode']
    data = []
    start_time = time.perf_counter()
    
    # Collect data for duration_s
    while (time.perf_counter() - start_time) < duration_s:
        time_stamp  = time.perf_counter() - start_time
        alpha_temp = (await eef.GetAnalogInput(alpha_feedback['alphaTempIn']))[0]
        # alpha_temp = al_gettemp()
        alpha_pd = (await eef.GetAnalogInput(alpha_feedback['alphaPhotoDiode']))[0]
        data.append(f"{time_stamp};{alpha_temp};{alpha_pd}\n")

    return(data)
